// Copyright 2025 Google LLC.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

// Code generated file. DO NOT EDIT.

// Package searchconsole provides access to the Google Search Console API.
//
// For product documentation, see: https://developers.google.com/webmaster-tools/about
//
// # Library status
//
// These client libraries are officially supported by Google. However, this
// library is considered complete and is in maintenance mode. This means
// that we will address critical bugs and security issues but will not add
// any new features.
//
// When possible, we recommend using our newer
// [Cloud Client Libraries for Go](https://pkg.go.dev/cloud.google.com/go)
// that are still actively being worked and iterated on.
//
// # Creating a client
//
// Usage example:
//
//	import "google.golang.org/api/searchconsole/v1"
//	...
//	ctx := context.Background()
//	searchconsoleService, err := searchconsole.NewService(ctx)
//
// In this example, Google Application Default Credentials are used for
// authentication. For information on how to create and obtain Application
// Default Credentials, see https://developers.google.com/identity/protocols/application-default-credentials.
//
// # Other authentication options
//
// By default, all available scopes (see "Constants") are used to authenticate.
// To restrict scopes, use [google.golang.org/api/option.WithScopes]:
//
//	searchconsoleService, err := searchconsole.NewService(ctx, option.WithScopes(searchconsole.WebmastersReadonlyScope))
//
// To use an API key for authentication (note: some APIs do not support API
// keys), use [google.golang.org/api/option.WithAPIKey]:
//
//	searchconsoleService, err := searchconsole.NewService(ctx, option.WithAPIKey("AIza..."))
//
// To use an OAuth token (e.g., a user token obtained via a three-legged OAuth
// flow, use [google.golang.org/api/option.WithTokenSource]:
//
//	config := &oauth2.Config{...}
//	// ...
//	token, err := config.Exchange(ctx, ...)
//	searchconsoleService, err := searchconsole.NewService(ctx, option.WithTokenSource(config.TokenSource(ctx, token)))
//
// See [google.golang.org/api/option.ClientOption] for details on options.
package searchconsole // import "google.golang.org/api/searchconsole/v1"

import (
	"bytes"
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"io"
	"log/slog"
	"net/http"
	"net/url"
	"strconv"
	"strings"

	"github.com/googleapis/gax-go/v2/internallog"
	googleapi "google.golang.org/api/googleapi"
	internal "google.golang.org/api/internal"
	gensupport "google.golang.org/api/internal/gensupport"
	option "google.golang.org/api/option"
	internaloption "google.golang.org/api/option/internaloption"
	htransport "google.golang.org/api/transport/http"
)

// Always reference these packages, just in case the auto-generated code
// below doesn't.
var _ = bytes.NewBuffer
var _ = strconv.Itoa
var _ = fmt.Sprintf
var _ = json.NewDecoder
var _ = io.Copy
var _ = url.Parse
var _ = gensupport.MarshalJSON
var _ = googleapi.Version
var _ = errors.New
var _ = strings.Replace
var _ = context.Canceled
var _ = internaloption.WithDefaultEndpoint
var _ = internal.Version
var _ = internallog.New

const apiId = "searchconsole:v1"
const apiName = "searchconsole"
const apiVersion = "v1"
const basePath = "https://searchconsole.googleapis.com/"
const basePathTemplate = "https://searchconsole.UNIVERSE_DOMAIN/"
const mtlsBasePath = "https://searchconsole.mtls.googleapis.com/"

// OAuth2 scopes used by this API.
const (
	// View and manage Search Console data for your verified sites
	WebmastersScope = "https://www.googleapis.com/auth/webmasters"

	// View Search Console data for your verified sites
	WebmastersReadonlyScope = "https://www.googleapis.com/auth/webmasters.readonly"
)

// NewService creates a new Service.
func NewService(ctx context.Context, opts ...option.ClientOption) (*Service, error) {
	scopesOption := internaloption.WithDefaultScopes(
		"https://www.googleapis.com/auth/webmasters",
		"https://www.googleapis.com/auth/webmasters.readonly",
	)
	// NOTE: prepend, so we don't override user-specified scopes.
	opts = append([]option.ClientOption{scopesOption}, opts...)
	opts = append(opts, internaloption.WithDefaultEndpoint(basePath))
	opts = append(opts, internaloption.WithDefaultEndpointTemplate(basePathTemplate))
	opts = append(opts, internaloption.WithDefaultMTLSEndpoint(mtlsBasePath))
	opts = append(opts, internaloption.EnableNewAuthLibrary())
	client, endpoint, err := htransport.NewClient(ctx, opts...)
	if err != nil {
		return nil, err
	}
	s := &Service{client: client, BasePath: basePath, logger: internaloption.GetLogger(opts)}
	s.Searchanalytics = NewSearchanalyticsService(s)
	s.Sitemaps = NewSitemapsService(s)
	s.Sites = NewSitesService(s)
	s.UrlInspection = NewUrlInspectionService(s)
	s.UrlTestingTools = NewUrlTestingToolsService(s)
	if endpoint != "" {
		s.BasePath = endpoint
	}
	return s, nil
}

// New creates a new Service. It uses the provided http.Client for requests.
//
// Deprecated: please use NewService instead.
// To provide a custom HTTP client, use option.WithHTTPClient.
// If you are using google.golang.org/api/googleapis/transport.APIKey, use option.WithAPIKey with NewService instead.
func New(client *http.Client) (*Service, error) {
	if client == nil {
		return nil, errors.New("client is nil")
	}
	return NewService(context.TODO(), option.WithHTTPClient(client))
}

type Service struct {
	client    *http.Client
	logger    *slog.Logger
	BasePath  string // API endpoint base URL
	UserAgent string // optional additional User-Agent fragment

	Searchanalytics *SearchanalyticsService

	Sitemaps *SitemapsService

	Sites *SitesService

	UrlInspection *UrlInspectionService

	UrlTestingTools *UrlTestingToolsService
}

func (s *Service) userAgent() string {
	if s.UserAgent == "" {
		return googleapi.UserAgent
	}
	return googleapi.UserAgent + " " + s.UserAgent
}

func NewSearchanalyticsService(s *Service) *SearchanalyticsService {
	rs := &SearchanalyticsService{s: s}
	return rs
}

type SearchanalyticsService struct {
	s *Service
}

func NewSitemapsService(s *Service) *SitemapsService {
	rs := &SitemapsService{s: s}
	return rs
}

type SitemapsService struct {
	s *Service
}

func NewSitesService(s *Service) *SitesService {
	rs := &SitesService{s: s}
	return rs
}

type SitesService struct {
	s *Service
}

func NewUrlInspectionService(s *Service) *UrlInspectionService {
	rs := &UrlInspectionService{s: s}
	rs.Index = NewUrlInspectionIndexService(s)
	return rs
}

type UrlInspectionService struct {
	s *Service

	Index *UrlInspectionIndexService
}

func NewUrlInspectionIndexService(s *Service) *UrlInspectionIndexService {
	rs := &UrlInspectionIndexService{s: s}
	return rs
}

type UrlInspectionIndexService struct {
	s *Service
}

func NewUrlTestingToolsService(s *Service) *UrlTestingToolsService {
	rs := &UrlTestingToolsService{s: s}
	rs.MobileFriendlyTest = NewUrlTestingToolsMobileFriendlyTestService(s)
	return rs
}

type UrlTestingToolsService struct {
	s *Service

	MobileFriendlyTest *UrlTestingToolsMobileFriendlyTestService
}

func NewUrlTestingToolsMobileFriendlyTestService(s *Service) *UrlTestingToolsMobileFriendlyTestService {
	rs := &UrlTestingToolsMobileFriendlyTestService{s: s}
	return rs
}

type UrlTestingToolsMobileFriendlyTestService struct {
	s *Service
}

// AmpInspectionResult: AMP inspection result of the live page or the current
// information from Google's index, depending on whether you requested a live
// inspection or not.
type AmpInspectionResult struct {
	// AmpIndexStatusVerdict: Index status of the AMP URL.
	//
	// Possible values:
	//   "VERDICT_UNSPECIFIED" - Unknown verdict.
	//   "PASS" - Equivalent to "Valid" for the page or item in Search Console.
	//   "PARTIAL" - Reserved, no longer in use.
	//   "FAIL" - Equivalent to "Error" or "Invalid" for the page or item in Search
	// Console.
	//   "NEUTRAL" - Equivalent to "Excluded" for the page or item in Search
	// Console.
	AmpIndexStatusVerdict string `json:"ampIndexStatusVerdict,omitempty"`
	// AmpUrl: URL of the AMP that was inspected. If the submitted URL is a desktop
	// page that refers to an AMP version, the AMP version will be inspected.
	AmpUrl string `json:"ampUrl,omitempty"`
	// IndexingState: Whether or not the page blocks indexing through a noindex
	// rule.
	//
	// Possible values:
	//   "AMP_INDEXING_STATE_UNSPECIFIED" - Unknown indexing status.
	//   "AMP_INDEXING_ALLOWED" - Indexing allowed.
	//   "BLOCKED_DUE_TO_NOINDEX" - Indexing not allowed, 'noindex' detected.
	//   "BLOCKED_DUE_TO_EXPIRED_UNAVAILABLE_AFTER" - Indexing not allowed,
	// 'unavailable_after' date expired.
	IndexingState string `json:"indexingState,omitempty"`
	// Issues: A list of zero or more AMP issues found for the inspected URL.
	Issues []*AmpIssue `json:"issues,omitempty"`
	// LastCrawlTime: Last time this AMP version was crawled by Google. Absent if
	// the URL was never crawled successfully.
	LastCrawlTime string `json:"lastCrawlTime,omitempty"`
	// PageFetchState: Whether or not Google could fetch the AMP.
	//
	// Possible values:
	//   "PAGE_FETCH_STATE_UNSPECIFIED" - Unknown fetch state.
	//   "SUCCESSFUL" - Successful fetch.
	//   "SOFT_404" - Soft 404.
	//   "BLOCKED_ROBOTS_TXT" - Blocked by robots.txt.
	//   "NOT_FOUND" - Not found (404).
	//   "ACCESS_DENIED" - Blocked due to unauthorized request (401).
	//   "SERVER_ERROR" - Server error (5xx).
	//   "REDIRECT_ERROR" - Redirection error.
	//   "ACCESS_FORBIDDEN" - Blocked due to access forbidden (403).
	//   "BLOCKED_4XX" - Blocked due to other 4xx issue (not 403, 404).
	//   "INTERNAL_CRAWL_ERROR" - Internal error.
	//   "INVALID_URL" - Invalid URL.
	PageFetchState string `json:"pageFetchState,omitempty"`
	// RobotsTxtState: Whether or not the page is blocked to Google by a robots.txt
	// rule.
	//
	// Possible values:
	//   "ROBOTS_TXT_STATE_UNSPECIFIED" - Unknown robots.txt state, typically
	// because the page wasn't fetched or found, or because robots.txt itself
	// couldn't be reached.
	//   "ALLOWED" - Crawl allowed by robots.txt.
	//   "DISALLOWED" - Crawl blocked by robots.txt.
	RobotsTxtState string `json:"robotsTxtState,omitempty"`
	// Verdict: The status of the most severe error on the page. If a page has both
	// warnings and errors, the page status is error. Error status means the page
	// cannot be shown in Search results.
	//
	// Possible values:
	//   "VERDICT_UNSPECIFIED" - Unknown verdict.
	//   "PASS" - Equivalent to "Valid" for the page or item in Search Console.
	//   "PARTIAL" - Reserved, no longer in use.
	//   "FAIL" - Equivalent to "Error" or "Invalid" for the page or item in Search
	// Console.
	//   "NEUTRAL" - Equivalent to "Excluded" for the page or item in Search
	// Console.
	Verdict string `json:"verdict,omitempty"`
	// ForceSendFields is a list of field names (e.g. "AmpIndexStatusVerdict") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AmpIndexStatusVerdict") to
	// include in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s AmpInspectionResult) MarshalJSON() ([]byte, error) {
	type NoMethod AmpInspectionResult
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// AmpIssue: AMP issue.
type AmpIssue struct {
	// IssueMessage: Brief description of this issue.
	IssueMessage string `json:"issueMessage,omitempty"`
	// Severity: Severity of this issue: WARNING or ERROR.
	//
	// Possible values:
	//   "SEVERITY_UNSPECIFIED" - Unknown severity.
	//   "WARNING" - Warning.
	//   "ERROR" - Error.
	Severity string `json:"severity,omitempty"`
	// ForceSendFields is a list of field names (e.g. "IssueMessage") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "IssueMessage") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s AmpIssue) MarshalJSON() ([]byte, error) {
	type NoMethod AmpIssue
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

type ApiDataRow struct {
	Clicks      float64  `json:"clicks,omitempty"`
	Ctr         float64  `json:"ctr,omitempty"`
	Impressions float64  `json:"impressions,omitempty"`
	Keys        []string `json:"keys,omitempty"`
	Position    float64  `json:"position,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Clicks") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Clicks") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ApiDataRow) MarshalJSON() ([]byte, error) {
	type NoMethod ApiDataRow
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

func (s *ApiDataRow) UnmarshalJSON(data []byte) error {
	type NoMethod ApiDataRow
	var s1 struct {
		Clicks      gensupport.JSONFloat64 `json:"clicks"`
		Ctr         gensupport.JSONFloat64 `json:"ctr"`
		Impressions gensupport.JSONFloat64 `json:"impressions"`
		Position    gensupport.JSONFloat64 `json:"position"`
		*NoMethod
	}
	s1.NoMethod = (*NoMethod)(s)
	if err := json.Unmarshal(data, &s1); err != nil {
		return err
	}
	s.Clicks = float64(s1.Clicks)
	s.Ctr = float64(s1.Ctr)
	s.Impressions = float64(s1.Impressions)
	s.Position = float64(s1.Position)
	return nil
}

// ApiDimensionFilter: A filter test to be applied to each row in the data set,
// where a match can return the row. Filters are string comparisons, and values
// and dimension names are not case-sensitive. Individual filters are either
// AND'ed or OR'ed within their parent filter group, according to the group's
// group type. You do not need to group by a specified dimension to filter
// against it.
type ApiDimensionFilter struct {
	// Possible values:
	//   "QUERY"
	//   "PAGE"
	//   "COUNTRY"
	//   "DEVICE"
	//   "SEARCH_APPEARANCE"
	Dimension  string `json:"dimension,omitempty"`
	Expression string `json:"expression,omitempty"`
	// Possible values:
	//   "EQUALS"
	//   "NOT_EQUALS"
	//   "CONTAINS"
	//   "NOT_CONTAINS"
	//   "INCLUDING_REGEX"
	//   "EXCLUDING_REGEX"
	Operator string `json:"operator,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Dimension") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Dimension") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ApiDimensionFilter) MarshalJSON() ([]byte, error) {
	type NoMethod ApiDimensionFilter
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ApiDimensionFilterGroup: A set of dimension value filters to test against
// each row. Only rows that pass all filter groups will be returned. All
// results within a filter group are either AND'ed or OR'ed together, depending
// on the group type selected. All filter groups are AND'ed together.
type ApiDimensionFilterGroup struct {
	Filters []*ApiDimensionFilter `json:"filters,omitempty"`
	// Possible values:
	//   "AND"
	GroupType string `json:"groupType,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Filters") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Filters") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ApiDimensionFilterGroup) MarshalJSON() ([]byte, error) {
	type NoMethod ApiDimensionFilterGroup
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// BlockedResource: Blocked resource.
type BlockedResource struct {
	// Url: URL of the blocked resource.
	Url string `json:"url,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Url") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Url") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s BlockedResource) MarshalJSON() ([]byte, error) {
	type NoMethod BlockedResource
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// DetectedItems: Rich Results items grouped by type.
type DetectedItems struct {
	// Items: List of Rich Results items.
	Items []*Item `json:"items,omitempty"`
	// RichResultType: Rich Results type
	RichResultType string `json:"richResultType,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Items") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Items") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s DetectedItems) MarshalJSON() ([]byte, error) {
	type NoMethod DetectedItems
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Image: Describe image data.
type Image struct {
	// Data: Image data in format determined by the mime type. Currently, the
	// format will always be "image/png", but this might change in the future.
	Data string `json:"data,omitempty"`
	// MimeType: The mime-type of the image data.
	MimeType string `json:"mimeType,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Data") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Data") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Image) MarshalJSON() ([]byte, error) {
	type NoMethod Image
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// IndexStatusInspectionResult: Results of index status inspection for either
// the live page or the version in Google's index, depending on whether you
// requested a live inspection or not. For more information, see the Index
// coverage report documentation
// (https://support.google.com/webmasters/answer/7440203).
type IndexStatusInspectionResult struct {
	// CoverageState: Could Google find and index the page. More details about page
	// indexing appear in 'indexing_state'.
	CoverageState string `json:"coverageState,omitempty"`
	// CrawledAs: Primary crawler that was used by Google to crawl your site.
	//
	// Possible values:
	//   "CRAWLING_USER_AGENT_UNSPECIFIED" - Unknown user agent.
	//   "DESKTOP" - Desktop user agent.
	//   "MOBILE" - Mobile user agent.
	CrawledAs string `json:"crawledAs,omitempty"`
	// GoogleCanonical: The URL of the page that Google selected as canonical. If
	// the page was not indexed, this field is absent.
	GoogleCanonical string `json:"googleCanonical,omitempty"`
	// IndexingState: Whether or not the page blocks indexing through a noindex
	// rule.
	//
	// Possible values:
	//   "INDEXING_STATE_UNSPECIFIED" - Unknown indexing status.
	//   "INDEXING_ALLOWED" - Indexing allowed.
	//   "BLOCKED_BY_META_TAG" - Indexing not allowed, 'noindex' detected in
	// 'robots' meta tag.
	//   "BLOCKED_BY_HTTP_HEADER" - Indexing not allowed, 'noindex' detected in
	// 'X-Robots-Tag' http header.
	//   "BLOCKED_BY_ROBOTS_TXT" - Reserved, no longer in use.
	IndexingState string `json:"indexingState,omitempty"`
	// LastCrawlTime: Last time this URL was crawled by Google using the primary
	// crawler
	// (https://support.google.com/webmasters/answer/7440203#primary_crawler).
	// Absent if the URL was never crawled successfully.
	LastCrawlTime string `json:"lastCrawlTime,omitempty"`
	// PageFetchState: Whether or not Google could retrieve the page from your
	// server. Equivalent to "page fetch"
	// (https://support.google.com/webmasters/answer/9012289#index_coverage) in the
	// URL inspection report.
	//
	// Possible values:
	//   "PAGE_FETCH_STATE_UNSPECIFIED" - Unknown fetch state.
	//   "SUCCESSFUL" - Successful fetch.
	//   "SOFT_404" - Soft 404.
	//   "BLOCKED_ROBOTS_TXT" - Blocked by robots.txt.
	//   "NOT_FOUND" - Not found (404).
	//   "ACCESS_DENIED" - Blocked due to unauthorized request (401).
	//   "SERVER_ERROR" - Server error (5xx).
	//   "REDIRECT_ERROR" - Redirection error.
	//   "ACCESS_FORBIDDEN" - Blocked due to access forbidden (403).
	//   "BLOCKED_4XX" - Blocked due to other 4xx issue (not 403, 404).
	//   "INTERNAL_CRAWL_ERROR" - Internal error.
	//   "INVALID_URL" - Invalid URL.
	PageFetchState string `json:"pageFetchState,omitempty"`
	// ReferringUrls: URLs that link to the inspected URL, directly and indirectly.
	ReferringUrls []string `json:"referringUrls,omitempty"`
	// RobotsTxtState: Whether or not the page is blocked to Google by a robots.txt
	// rule.
	//
	// Possible values:
	//   "ROBOTS_TXT_STATE_UNSPECIFIED" - Unknown robots.txt state, typically
	// because the page wasn't fetched or found, or because robots.txt itself
	// couldn't be reached.
	//   "ALLOWED" - Crawl allowed by robots.txt.
	//   "DISALLOWED" - Crawl blocked by robots.txt.
	RobotsTxtState string `json:"robotsTxtState,omitempty"`
	// Sitemap: Any sitemaps that this URL was listed in, as known by Google. Not
	// guaranteed to be an exhaustive list, especially if Google did not discover
	// this URL through a sitemap. Absent if no sitemaps were found.
	Sitemap []string `json:"sitemap,omitempty"`
	// UserCanonical: The URL that your page or site declares as canonical
	// (https://developers.google.com/search/docs/advanced/crawling/consolidate-duplicate-urls?#define-canonical).
	// If you did not declare a canonical URL, this field is absent.
	UserCanonical string `json:"userCanonical,omitempty"`
	// Verdict: High level verdict about whether the URL *is* indexed (indexed
	// status), or *can be* indexed (live inspection).
	//
	// Possible values:
	//   "VERDICT_UNSPECIFIED" - Unknown verdict.
	//   "PASS" - Equivalent to "Valid" for the page or item in Search Console.
	//   "PARTIAL" - Reserved, no longer in use.
	//   "FAIL" - Equivalent to "Error" or "Invalid" for the page or item in Search
	// Console.
	//   "NEUTRAL" - Equivalent to "Excluded" for the page or item in Search
	// Console.
	Verdict string `json:"verdict,omitempty"`
	// ForceSendFields is a list of field names (e.g. "CoverageState") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "CoverageState") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s IndexStatusInspectionResult) MarshalJSON() ([]byte, error) {
	type NoMethod IndexStatusInspectionResult
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// InspectUrlIndexRequest: Index inspection request.
type InspectUrlIndexRequest struct {
	// InspectionUrl: Required. URL to inspect. Must be under the property
	// specified in "site_url".
	InspectionUrl string `json:"inspectionUrl,omitempty"`
	// LanguageCode: Optional. An IETF BCP-47
	// (https://en.wikipedia.org/wiki/IETF_language_tag) language code representing
	// the requested language for translated issue messages, e.g. "en-US", "or
	// "de-CH". Default value is "en-US".
	LanguageCode string `json:"languageCode,omitempty"`
	// SiteUrl: Required. The URL of the property as defined in Search Console.
	// **Examples:** `http://www.example.com/` for a URL-prefix property, or
	// `sc-domain:example.com` for a Domain property.
	SiteUrl string `json:"siteUrl,omitempty"`
	// ForceSendFields is a list of field names (e.g. "InspectionUrl") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "InspectionUrl") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s InspectUrlIndexRequest) MarshalJSON() ([]byte, error) {
	type NoMethod InspectUrlIndexRequest
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// InspectUrlIndexResponse: Index-Status inspection response.
type InspectUrlIndexResponse struct {
	// InspectionResult: URL inspection results.
	InspectionResult *UrlInspectionResult `json:"inspectionResult,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "InspectionResult") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "InspectionResult") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s InspectUrlIndexResponse) MarshalJSON() ([]byte, error) {
	type NoMethod InspectUrlIndexResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Item: A specific rich result found on the page.
type Item struct {
	// Issues: A list of zero or more rich result issues found for this instance.
	Issues []*RichResultsIssue `json:"issues,omitempty"`
	// Name: The user-provided name of this item.
	Name string `json:"name,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Issues") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Issues") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Item) MarshalJSON() ([]byte, error) {
	type NoMethod Item
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Metadata: An object that may be returned with your query results, providing
// context about the state of the data. When you request recent data (using
// `all` or `hourly_all` for `dataState`), some of the rows returned may
// represent data that is incomplete, which means that the data is still being
// collected and processed. This metadata object helps you identify exactly
// when this starts and ends. All dates and times provided in this object are
// in the `America/Los_Angeles` time zone. The specific field returned within
// this object depends on how you've grouped your data in the request. See
// details in inner fields.
type Metadata struct {
	// FirstIncompleteDate: The first date for which the data is still being
	// collected and processed, presented in `YYYY-MM-DD` format (ISO-8601 extended
	// local date format). This field is populated only when the request's
	// `dataState` is "all", data is grouped by "DATE", and the requested date
	// range contains incomplete data points. All values after the
	// `first_incomplete_date` may still change noticeably.
	FirstIncompleteDate string `json:"firstIncompleteDate,omitempty"`
	// FirstIncompleteHour: The first hour for which the data is still being
	// collected and processed, presented in `YYYY-MM-DDThh:mm:ss[+|-]hh:mm` format
	// (ISO-8601 extended offset date-time format). This field is populated only
	// when the request's `dataState` is "hourly_all", data is grouped by
	// "HOUR" and the requested date range contains incomplete data points. All
	// values after the `first_incomplete_hour` may still change noticeably.
	FirstIncompleteHour string `json:"firstIncompleteHour,omitempty"`
	// ForceSendFields is a list of field names (e.g. "FirstIncompleteDate") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "FirstIncompleteDate") to include
	// in API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Metadata) MarshalJSON() ([]byte, error) {
	type NoMethod Metadata
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// MobileFriendlyIssue: Mobile-friendly issue.
type MobileFriendlyIssue struct {
	// Rule: Rule violated.
	//
	// Possible values:
	//   "MOBILE_FRIENDLY_RULE_UNSPECIFIED" - Unknown rule. Sorry, we don't have
	// any description for the rule that was broken.
	//   "USES_INCOMPATIBLE_PLUGINS" - Plugins incompatible with mobile devices are
	// being used. [Learn more]
	// (https://support.google.com/webmasters/answer/6352293#flash_usage).
	//   "CONFIGURE_VIEWPORT" - Viewport is not specified using the meta viewport
	// tag. [Learn more]
	// (https://support.google.com/webmasters/answer/6352293#viewport_not_configured).
	//   "FIXED_WIDTH_VIEWPORT" - Viewport defined to a fixed width. [Learn more]
	// (https://support.google.com/webmasters/answer/6352293#fixed-width_viewport).
	//   "SIZE_CONTENT_TO_VIEWPORT" - Content not sized to viewport. [Learn more]
	// (https://support.google.com/webmasters/answer/6352293#content_not_sized_to_viewport).
	//   "USE_LEGIBLE_FONT_SIZES" - Font size is too small for easy reading on a
	// small screen. [Learn More]
	// (https://support.google.com/webmasters/answer/6352293#small_font_size).
	//   "TAP_TARGETS_TOO_CLOSE" - Touch elements are too close to each other.
	// [Learn more]
	// (https://support.google.com/webmasters/answer/6352293#touch_elements_too_close).
	Rule string `json:"rule,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Rule") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Rule") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s MobileFriendlyIssue) MarshalJSON() ([]byte, error) {
	type NoMethod MobileFriendlyIssue
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// MobileUsabilityInspectionResult: Mobile-usability inspection results.
type MobileUsabilityInspectionResult struct {
	// Issues: A list of zero or more mobile-usability issues detected for this
	// URL.
	Issues []*MobileUsabilityIssue `json:"issues,omitempty"`
	// Verdict: High-level mobile-usability inspection result for this URL.
	//
	// Possible values:
	//   "VERDICT_UNSPECIFIED" - Unknown verdict.
	//   "PASS" - Equivalent to "Valid" for the page or item in Search Console.
	//   "PARTIAL" - Reserved, no longer in use.
	//   "FAIL" - Equivalent to "Error" or "Invalid" for the page or item in Search
	// Console.
	//   "NEUTRAL" - Equivalent to "Excluded" for the page or item in Search
	// Console.
	Verdict string `json:"verdict,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Issues") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Issues") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s MobileUsabilityInspectionResult) MarshalJSON() ([]byte, error) {
	type NoMethod MobileUsabilityInspectionResult
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// MobileUsabilityIssue: Mobile-usability issue.
type MobileUsabilityIssue struct {
	// IssueType: Mobile-usability issue type.
	//
	// Possible values:
	//   "MOBILE_USABILITY_ISSUE_TYPE_UNSPECIFIED" - Unknown issue. Sorry, we don't
	// have any description for the rule that was broken.
	//   "USES_INCOMPATIBLE_PLUGINS" - Plugins incompatible with mobile devices are
	// being used. [Learn more]
	// (https://support.google.com/webmasters/answer/6352293#flash_usage#error-list).
	//   "CONFIGURE_VIEWPORT" - Viewport is not specified using the meta viewport
	// tag. [Learn more]
	// (https://support.google.com/webmasters/answer/6352293#viewport_not_configured#error-list).
	//   "FIXED_WIDTH_VIEWPORT" - Viewport defined to a fixed width. [Learn more]
	// (https://support.google.com/webmasters/answer/6352293#fixed-width_viewport#error-list).
	//   "SIZE_CONTENT_TO_VIEWPORT" - Content not sized to viewport. [Learn more]
	// (https://support.google.com/webmasters/answer/6352293#content_not_sized_to_viewport#error-list).
	//   "USE_LEGIBLE_FONT_SIZES" - Font size is too small for easy reading on a
	// small screen. [Learn More]
	// (https://support.google.com/webmasters/answer/6352293#small_font_size#error-list).
	//   "TAP_TARGETS_TOO_CLOSE" - Touch elements are too close to each other.
	// [Learn more]
	// (https://support.google.com/webmasters/answer/6352293#touch_elements_too_close#error-list).
	IssueType string `json:"issueType,omitempty"`
	// Message: Additional information regarding the issue.
	Message string `json:"message,omitempty"`
	// Severity: Not returned; reserved for future use.
	//
	// Possible values:
	//   "SEVERITY_UNSPECIFIED" - Unknown severity.
	//   "WARNING" - Warning.
	//   "ERROR" - Error.
	Severity string `json:"severity,omitempty"`
	// ForceSendFields is a list of field names (e.g. "IssueType") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "IssueType") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s MobileUsabilityIssue) MarshalJSON() ([]byte, error) {
	type NoMethod MobileUsabilityIssue
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ResourceIssue: Information about a resource with issue.
type ResourceIssue struct {
	// BlockedResource: Describes a blocked resource issue.
	BlockedResource *BlockedResource `json:"blockedResource,omitempty"`
	// ForceSendFields is a list of field names (e.g. "BlockedResource") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "BlockedResource") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ResourceIssue) MarshalJSON() ([]byte, error) {
	type NoMethod ResourceIssue
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// RichResultsInspectionResult: Rich-Results inspection result, including any
// rich results found at this URL.
type RichResultsInspectionResult struct {
	// DetectedItems: A list of zero or more rich results detected on this page.
	// Rich results that cannot even be parsed due to syntactic issues will not be
	// listed here.
	DetectedItems []*DetectedItems `json:"detectedItems,omitempty"`
	// Verdict: High-level rich results inspection result for this URL.
	//
	// Possible values:
	//   "VERDICT_UNSPECIFIED" - Unknown verdict.
	//   "PASS" - Equivalent to "Valid" for the page or item in Search Console.
	//   "PARTIAL" - Reserved, no longer in use.
	//   "FAIL" - Equivalent to "Error" or "Invalid" for the page or item in Search
	// Console.
	//   "NEUTRAL" - Equivalent to "Excluded" for the page or item in Search
	// Console.
	Verdict string `json:"verdict,omitempty"`
	// ForceSendFields is a list of field names (e.g. "DetectedItems") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "DetectedItems") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s RichResultsInspectionResult) MarshalJSON() ([]byte, error) {
	type NoMethod RichResultsInspectionResult
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// RichResultsIssue: Severity and status of a single issue affecting a single
// rich result instance on a page.
type RichResultsIssue struct {
	// IssueMessage: Rich Results issue type.
	IssueMessage string `json:"issueMessage,omitempty"`
	// Severity: Severity of this issue: WARNING, or ERROR. Items with an issue of
	// status ERROR cannot appear with rich result features in Google Search
	// results.
	//
	// Possible values:
	//   "SEVERITY_UNSPECIFIED" - Unknown severity.
	//   "WARNING" - Warning.
	//   "ERROR" - Error.
	Severity string `json:"severity,omitempty"`
	// ForceSendFields is a list of field names (e.g. "IssueMessage") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "IssueMessage") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s RichResultsIssue) MarshalJSON() ([]byte, error) {
	type NoMethod RichResultsIssue
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// RunMobileFriendlyTestRequest: Mobile-friendly test request.
type RunMobileFriendlyTestRequest struct {
	// RequestScreenshot: Whether or not screenshot is requested. Default is false.
	RequestScreenshot bool `json:"requestScreenshot,omitempty"`
	// Url: URL for inspection.
	Url string `json:"url,omitempty"`
	// ForceSendFields is a list of field names (e.g. "RequestScreenshot") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "RequestScreenshot") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s RunMobileFriendlyTestRequest) MarshalJSON() ([]byte, error) {
	type NoMethod RunMobileFriendlyTestRequest
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// RunMobileFriendlyTestResponse: Mobile-friendly test response, including
// mobile-friendly issues and resource issues.
type RunMobileFriendlyTestResponse struct {
	// MobileFriendliness: Test verdict, whether the page is mobile friendly or
	// not.
	//
	// Possible values:
	//   "MOBILE_FRIENDLY_TEST_RESULT_UNSPECIFIED" - Internal error when running
	// this test. Please try running the test again.
	//   "MOBILE_FRIENDLY" - The page is mobile friendly.
	//   "NOT_MOBILE_FRIENDLY" - The page is not mobile friendly.
	MobileFriendliness string `json:"mobileFriendliness,omitempty"`
	// MobileFriendlyIssues: List of mobile-usability issues.
	MobileFriendlyIssues []*MobileFriendlyIssue `json:"mobileFriendlyIssues,omitempty"`
	// ResourceIssues: Information about embedded resources issues.
	ResourceIssues []*ResourceIssue `json:"resourceIssues,omitempty"`
	// Screenshot: Screenshot of the requested URL.
	Screenshot *Image `json:"screenshot,omitempty"`
	// TestStatus: Final state of the test, can be either complete or an error.
	TestStatus *TestStatus `json:"testStatus,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "MobileFriendliness") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "MobileFriendliness") to include
	// in API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s RunMobileFriendlyTestResponse) MarshalJSON() ([]byte, error) {
	type NoMethod RunMobileFriendlyTestResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

type SearchAnalyticsQueryRequest struct {
	// AggregationType: [Optional; Default is \"auto\"] How data is aggregated. If
	// aggregated by property, all data for the same property is aggregated; if
	// aggregated by page, all data is aggregated by canonical URI. If you filter
	// or group by page, choose AUTO; otherwise you can aggregate either by
	// property or by page, depending on how you want your data calculated; see the
	// help documentation to learn how data is calculated differently by site
	// versus by page. **Note:** If you group or filter by page, you cannot
	// aggregate by property. If you specify any value other than AUTO, the
	// aggregation type in the result will match the requested type, or if you
	// request an invalid type, you will get an error. The API will never change
	// your aggregation type if the requested type is invalid.
	//
	// Possible values:
	//   "AUTO"
	//   "BY_PROPERTY"
	//   "BY_PAGE"
	//   "BY_NEWS_SHOWCASE_PANEL"
	AggregationType string `json:"aggregationType,omitempty"`
	// DataState: The data state to be fetched, can be full or all, the latter
	// including full and partial data.
	//
	// Possible values:
	//   "DATA_STATE_UNSPECIFIED" - Default value, should not be used.
	//   "FINAL" - Include full final data only, without partial.
	//   "ALL" - Include all data, full and partial.
	//   "HOURLY_ALL" - Include hourly data, full and partial. Required when
	// grouping by HOUR.
	DataState string `json:"dataState,omitempty"`
	// DimensionFilterGroups: [Optional] Zero or more filters to apply to the
	// dimension grouping values; for example, 'query contains \"buy\"' to see only
	// data where the query string contains the substring \"buy\" (not
	// case-sensitive). You can filter by a dimension without grouping by it.
	DimensionFilterGroups []*ApiDimensionFilterGroup `json:"dimensionFilterGroups,omitempty"`
	// Dimensions: [Optional] Zero or more dimensions to group results by.
	// Dimensions are the group-by values in the Search Analytics page. Dimensions
	// are combined to create a unique row key for each row. Results are grouped in
	// the order that you supply these dimensions.
	//
	// Possible values:
	//   "DATE" - Group by date, which is returned in YYYY-MM-DD format, in PT time
	// (UTC - 7:00/8:00).
	//   "QUERY" - Group by query string.
	//   "PAGE" - Group by page, a URI string.
	//   "COUNTRY" - Group by country, specified by 3-letter country code ([ISO
	// 3166-1 alpha-3](https://en.wikipedia.org/wiki/ISO_3166-1_alpha-3)).
	//   "DEVICE" - Group by device type (DESKTOP, MOBILE and TABLET).
	//   "SEARCH_APPEARANCE" - Group by search result feature.
	//   "HOUR" - Group by hour, which is returned in YYYY-MM-DDThh:mm:ss[+|-]hh:mm
	// format (ISO-8601 extended offset date-time format) in PT time (UTC -
	// 7:00/8:00). Data is available up to 10 days. Requires setting the dataState
	// to HOURLY_ALL.
	Dimensions []string `json:"dimensions,omitempty"`
	// EndDate: [Required] End date of the requested date range, in YYYY-MM-DD
	// format, in PST (UTC - 8:00). Must be greater than or equal to the start
	// date. This value is included in the range.
	EndDate string `json:"endDate,omitempty"`
	// RowLimit: [Optional; Default is 1000] The maximum number of rows to return.
	// Must be a number from 1 to 25,000 (inclusive).
	RowLimit int64 `json:"rowLimit,omitempty"`
	// SearchType: [Optional; Default is \"web\"] The search type to filter for.
	//
	// Possible values:
	//   "WEB"
	//   "IMAGE"
	//   "VIDEO"
	//   "NEWS" - News tab in search.
	//   "DISCOVER" - Discover.
	//   "GOOGLE_NEWS" - Google News (news.google.com or mobile app).
	SearchType string `json:"searchType,omitempty"`
	// StartDate:  [Required] Start date of the requested date range, in YYYY-MM-DD
	// format, in PST time (UTC - 8:00). Must be less than or equal to the end
	// date. This value is included in the range.
	StartDate string `json:"startDate,omitempty"`
	// StartRow: [Optional; Default is 0] Zero-based index of the first row in the
	// response. Must be a non-negative number.
	StartRow int64 `json:"startRow,omitempty"`
	// Type: Optional. [Optional; Default is \"web\"] Type of report: search type,
	// or either Discover or Gnews.
	//
	// Possible values:
	//   "WEB"
	//   "IMAGE"
	//   "VIDEO"
	//   "NEWS" - News tab in search.
	//   "DISCOVER" - Discover.
	//   "GOOGLE_NEWS" - Google News (news.google.com or mobile app).
	Type string `json:"type,omitempty"`
	// ForceSendFields is a list of field names (e.g. "AggregationType") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AggregationType") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s SearchAnalyticsQueryRequest) MarshalJSON() ([]byte, error) {
	type NoMethod SearchAnalyticsQueryRequest
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// SearchAnalyticsQueryResponse: A list of rows, one per result, grouped by
// key. Metrics in each row are aggregated for all data grouped by that key
// either by page or property, as specified by the aggregation type parameter.
type SearchAnalyticsQueryResponse struct {
	// Metadata: An object that may be returned with your query results, providing
	// context about the state of the data. See details in Metadata object
	// documentation.
	Metadata *Metadata `json:"metadata,omitempty"`
	// ResponseAggregationType: How the results were aggregated.
	//
	// Possible values:
	//   "AUTO"
	//   "BY_PROPERTY"
	//   "BY_PAGE"
	//   "BY_NEWS_SHOWCASE_PANEL"
	ResponseAggregationType string `json:"responseAggregationType,omitempty"`
	// Rows: A list of rows grouped by the key values in the order given in the
	// query.
	Rows []*ApiDataRow `json:"rows,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Metadata") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Metadata") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s SearchAnalyticsQueryResponse) MarshalJSON() ([]byte, error) {
	type NoMethod SearchAnalyticsQueryResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// SitemapsListResponse: List of sitemaps.
type SitemapsListResponse struct {
	// Sitemap: Contains detailed information about a specific URL submitted as a
	// sitemap (https://support.google.com/webmasters/answer/156184).
	Sitemap []*WmxSitemap `json:"sitemap,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Sitemap") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Sitemap") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s SitemapsListResponse) MarshalJSON() ([]byte, error) {
	type NoMethod SitemapsListResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// SitesListResponse: List of sites with access level information.
type SitesListResponse struct {
	// SiteEntry: Contains permission level information about a Search Console
	// site. For more information, see Permissions in Search Console
	// (https://support.google.com/webmasters/answer/2451999).
	SiteEntry []*WmxSite `json:"siteEntry,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "SiteEntry") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "SiteEntry") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s SitesListResponse) MarshalJSON() ([]byte, error) {
	type NoMethod SitesListResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// TestStatus: Final state of the test, including error details if necessary.
type TestStatus struct {
	// Details: Error details if applicable.
	Details string `json:"details,omitempty"`
	// Status: Status of the test.
	//
	// Possible values:
	//   "TEST_STATUS_UNSPECIFIED" - Internal error when running this test. Please
	// try running the test again.
	//   "COMPLETE" - Inspection has completed without errors.
	//   "INTERNAL_ERROR" - Inspection terminated in an error state. This indicates
	// a problem in Google's infrastructure, not a user error. Please try again
	// later.
	//   "PAGE_UNREACHABLE" - Google can not access the URL because of a user error
	// such as a robots.txt blockage, a 403 or 500 code etc. Please make sure that
	// the URL provided is accessible by Googlebot and is not password protected.
	Status string `json:"status,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Details") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Details") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s TestStatus) MarshalJSON() ([]byte, error) {
	type NoMethod TestStatus
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// UrlInspectionResult: URL inspection result, including all inspection
// results.
type UrlInspectionResult struct {
	// AmpResult: Result of the AMP analysis. Absent if the page is not an AMP
	// page.
	AmpResult *AmpInspectionResult `json:"ampResult,omitempty"`
	// IndexStatusResult: Result of the index status analysis.
	IndexStatusResult *IndexStatusInspectionResult `json:"indexStatusResult,omitempty"`
	// InspectionResultLink: Link to Search Console URL inspection.
	InspectionResultLink string `json:"inspectionResultLink,omitempty"`
	// MobileUsabilityResult: Result of the Mobile usability analysis.
	MobileUsabilityResult *MobileUsabilityInspectionResult `json:"mobileUsabilityResult,omitempty"`
	// RichResultsResult: Result of the Rich Results analysis. Absent if there are
	// no rich results found.
	RichResultsResult *RichResultsInspectionResult `json:"richResultsResult,omitempty"`
	// ForceSendFields is a list of field names (e.g. "AmpResult") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AmpResult") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s UrlInspectionResult) MarshalJSON() ([]byte, error) {
	type NoMethod UrlInspectionResult
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// WmxSite: Contains permission level information about a Search Console site.
// For more information, see Permissions in Search Console
// (https://support.google.com/webmasters/answer/2451999).
type WmxSite struct {
	// PermissionLevel: The user's permission level for the site.
	//
	// Possible values:
	//   "SITE_PERMISSION_LEVEL_UNSPECIFIED"
	//   "SITE_OWNER" - Owner has complete access to the site.
	//   "SITE_FULL_USER" - Full users can access all data, and perform most of the
	// operations.
	//   "SITE_RESTRICTED_USER" - Restricted users can access most of the data, and
	// perform some operations.
	//   "SITE_UNVERIFIED_USER" - Unverified user has no access to site's data.
	PermissionLevel string `json:"permissionLevel,omitempty"`
	// SiteUrl: The URL of the site.
	SiteUrl string `json:"siteUrl,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "PermissionLevel") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "PermissionLevel") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s WmxSite) MarshalJSON() ([]byte, error) {
	type NoMethod WmxSite
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// WmxSitemap: Contains detailed information about a specific URL submitted as
// a sitemap (https://support.google.com/webmasters/answer/156184).
type WmxSitemap struct {
	// Contents: The various content types in the sitemap.
	Contents []*WmxSitemapContent `json:"contents,omitempty"`
	// Errors: Number of errors in the sitemap. These are issues with the sitemap
	// itself that need to be fixed before it can be processed correctly.
	Errors int64 `json:"errors,omitempty,string"`
	// IsPending: If true, the sitemap has not been processed.
	IsPending bool `json:"isPending,omitempty"`
	// IsSitemapsIndex: If true, the sitemap is a collection of sitemaps.
	IsSitemapsIndex bool `json:"isSitemapsIndex,omitempty"`
	// LastDownloaded: Date & time in which this sitemap was last downloaded. Date
	// format is in RFC 3339 format (yyyy-mm-dd).
	LastDownloaded string `json:"lastDownloaded,omitempty"`
	// LastSubmitted: Date & time in which this sitemap was submitted. Date format
	// is in RFC 3339 format (yyyy-mm-dd).
	LastSubmitted string `json:"lastSubmitted,omitempty"`
	// Path: The url of the sitemap.
	Path string `json:"path,omitempty"`
	// Type: The type of the sitemap. For example: `rssFeed`.
	//
	// Possible values:
	//   "NOT_SITEMAP"
	//   "URL_LIST"
	//   "SITEMAP"
	//   "RSS_FEED"
	//   "ATOM_FEED"
	//   "PATTERN_SITEMAP" - Unsupported sitemap types.
	//   "OCEANFRONT"
	Type string `json:"type,omitempty"`
	// Warnings: Number of warnings for the sitemap. These are generally
	// non-critical issues with URLs in the sitemaps.
	Warnings int64 `json:"warnings,omitempty,string"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Contents") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Contents") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s WmxSitemap) MarshalJSON() ([]byte, error) {
	type NoMethod WmxSitemap
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// WmxSitemapContent: Information about the various content types in the
// sitemap.
type WmxSitemapContent struct {
	// Indexed: *Deprecated; do not use.*
	Indexed int64 `json:"indexed,omitempty,string"`
	// Submitted: The number of URLs in the sitemap (of the content type).
	Submitted int64 `json:"submitted,omitempty,string"`
	// Type: The specific type of content in this sitemap. For example: `web`.
	//
	// Possible values:
	//   "WEB"
	//   "IMAGE"
	//   "VIDEO"
	//   "NEWS"
	//   "MOBILE"
	//   "ANDROID_APP"
	//   "PATTERN" - Unsupported content type.
	//   "IOS_APP"
	//   "DATA_FEED_ELEMENT" - Unsupported content type.
	Type string `json:"type,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Indexed") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Indexed") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s WmxSitemapContent) MarshalJSON() ([]byte, error) {
	type NoMethod WmxSitemapContent
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

type SearchanalyticsQueryCall struct {
	s                           *Service
	siteUrl                     string
	searchanalyticsqueryrequest *SearchAnalyticsQueryRequest
	urlParams_                  gensupport.URLParams
	ctx_                        context.Context
	header_                     http.Header
}

// Query: Query your data with filters and parameters that you define. Returns
// zero or more rows grouped by the row keys that you define. You must define a
// date range of one or more days. When date is one of the group by values, any
// days without data are omitted from the result list. If you need to know
// which days have data, issue a broad date range query grouped by date for any
// metric, and see which day rows are returned.
//
//   - siteUrl: The site's URL, including protocol. For example:
//     `http://www.example.com/`.
func (r *SearchanalyticsService) Query(siteUrl string, searchanalyticsqueryrequest *SearchAnalyticsQueryRequest) *SearchanalyticsQueryCall {
	c := &SearchanalyticsQueryCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.siteUrl = siteUrl
	c.searchanalyticsqueryrequest = searchanalyticsqueryrequest
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *SearchanalyticsQueryCall) Fields(s ...googleapi.Field) *SearchanalyticsQueryCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *SearchanalyticsQueryCall) Context(ctx context.Context) *SearchanalyticsQueryCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *SearchanalyticsQueryCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *SearchanalyticsQueryCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.searchanalyticsqueryrequest)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "webmasters/v3/sites/{siteUrl}/searchAnalytics/query")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"siteUrl": c.siteUrl,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "webmasters.searchanalytics.query", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "webmasters.searchanalytics.query" call.
// Any non-2xx status code is an error. Response headers are in either
// *SearchAnalyticsQueryResponse.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *SearchanalyticsQueryCall) Do(opts ...googleapi.CallOption) (*SearchAnalyticsQueryResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &SearchAnalyticsQueryResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "webmasters.searchanalytics.query", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type SitemapsDeleteCall struct {
	s          *Service
	siteUrl    string
	feedpath   string
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Delete: Deletes a sitemap from the Sitemaps report. Does not stop Google
// from crawling this sitemap or the URLs that were previously crawled in the
// deleted sitemap.
//
//   - feedpath: The URL of the actual sitemap. For example:
//     `http://www.example.com/sitemap.xml`.
//   - siteUrl: The site's URL, including protocol. For example:
//     `http://www.example.com/`.
func (r *SitemapsService) Delete(siteUrl string, feedpath string) *SitemapsDeleteCall {
	c := &SitemapsDeleteCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.siteUrl = siteUrl
	c.feedpath = feedpath
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *SitemapsDeleteCall) Fields(s ...googleapi.Field) *SitemapsDeleteCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *SitemapsDeleteCall) Context(ctx context.Context) *SitemapsDeleteCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *SitemapsDeleteCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *SitemapsDeleteCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "webmasters/v3/sites/{siteUrl}/sitemaps/{feedpath}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("DELETE", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"siteUrl":  c.siteUrl,
		"feedpath": c.feedpath,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "webmasters.sitemaps.delete", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "webmasters.sitemaps.delete" call.
func (c *SitemapsDeleteCall) Do(opts ...googleapi.CallOption) error {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if err != nil {
		return err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return gensupport.WrapError(err)
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "webmasters.sitemaps.delete", "response", internallog.HTTPResponse(res, nil))
	return nil
}

type SitemapsGetCall struct {
	s            *Service
	siteUrl      string
	feedpath     string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Retrieves information about a specific sitemap.
//
//   - feedpath: The URL of the actual sitemap. For example:
//     `http://www.example.com/sitemap.xml`.
//   - siteUrl: The site's URL, including protocol. For example:
//     `http://www.example.com/`.
func (r *SitemapsService) Get(siteUrl string, feedpath string) *SitemapsGetCall {
	c := &SitemapsGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.siteUrl = siteUrl
	c.feedpath = feedpath
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *SitemapsGetCall) Fields(s ...googleapi.Field) *SitemapsGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *SitemapsGetCall) IfNoneMatch(entityTag string) *SitemapsGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *SitemapsGetCall) Context(ctx context.Context) *SitemapsGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *SitemapsGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *SitemapsGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "webmasters/v3/sites/{siteUrl}/sitemaps/{feedpath}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"siteUrl":  c.siteUrl,
		"feedpath": c.feedpath,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "webmasters.sitemaps.get", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "webmasters.sitemaps.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *WmxSitemap.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *SitemapsGetCall) Do(opts ...googleapi.CallOption) (*WmxSitemap, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &WmxSitemap{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "webmasters.sitemaps.get", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type SitemapsListCall struct {
	s            *Service
	siteUrl      string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List:  Lists the sitemaps-entries (/webmaster-tools/v3/sitemaps) submitted
// for this site, or included in the sitemap index file (if `sitemapIndex` is
// specified in the request).
//
//   - siteUrl: The site's URL, including protocol. For example:
//     `http://www.example.com/`.
func (r *SitemapsService) List(siteUrl string) *SitemapsListCall {
	c := &SitemapsListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.siteUrl = siteUrl
	return c
}

// SitemapIndex sets the optional parameter "sitemapIndex": A URL of a site's
// sitemap index. For example: `http://www.example.com/sitemapindex.xml`.
func (c *SitemapsListCall) SitemapIndex(sitemapIndex string) *SitemapsListCall {
	c.urlParams_.Set("sitemapIndex", sitemapIndex)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *SitemapsListCall) Fields(s ...googleapi.Field) *SitemapsListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *SitemapsListCall) IfNoneMatch(entityTag string) *SitemapsListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *SitemapsListCall) Context(ctx context.Context) *SitemapsListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *SitemapsListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *SitemapsListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "webmasters/v3/sites/{siteUrl}/sitemaps")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"siteUrl": c.siteUrl,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "webmasters.sitemaps.list", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "webmasters.sitemaps.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *SitemapsListResponse.ServerResponse.Header or (if a response was returned
// at all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *SitemapsListCall) Do(opts ...googleapi.CallOption) (*SitemapsListResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &SitemapsListResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "webmasters.sitemaps.list", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type SitemapsSubmitCall struct {
	s          *Service
	siteUrl    string
	feedpath   string
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Submit: Submits a sitemap for a site.
//
//   - feedpath: The URL of the actual sitemap. For example:
//     `http://www.example.com/sitemap.xml`.
//   - siteUrl: The site's URL, including protocol. For example:
//     `http://www.example.com/`.
func (r *SitemapsService) Submit(siteUrl string, feedpath string) *SitemapsSubmitCall {
	c := &SitemapsSubmitCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.siteUrl = siteUrl
	c.feedpath = feedpath
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *SitemapsSubmitCall) Fields(s ...googleapi.Field) *SitemapsSubmitCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *SitemapsSubmitCall) Context(ctx context.Context) *SitemapsSubmitCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *SitemapsSubmitCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *SitemapsSubmitCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "webmasters/v3/sites/{siteUrl}/sitemaps/{feedpath}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("PUT", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"siteUrl":  c.siteUrl,
		"feedpath": c.feedpath,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "webmasters.sitemaps.submit", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "webmasters.sitemaps.submit" call.
func (c *SitemapsSubmitCall) Do(opts ...googleapi.CallOption) error {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if err != nil {
		return err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return gensupport.WrapError(err)
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "webmasters.sitemaps.submit", "response", internallog.HTTPResponse(res, nil))
	return nil
}

type SitesAddCall struct {
	s          *Service
	siteUrl    string
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Add:  Adds a site to the set of the user's sites in Search Console.
//
// - siteUrl: The URL of the site to add.
func (r *SitesService) Add(siteUrl string) *SitesAddCall {
	c := &SitesAddCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.siteUrl = siteUrl
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *SitesAddCall) Fields(s ...googleapi.Field) *SitesAddCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *SitesAddCall) Context(ctx context.Context) *SitesAddCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *SitesAddCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *SitesAddCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "webmasters/v3/sites/{siteUrl}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("PUT", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"siteUrl": c.siteUrl,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "webmasters.sites.add", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "webmasters.sites.add" call.
func (c *SitesAddCall) Do(opts ...googleapi.CallOption) error {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if err != nil {
		return err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return gensupport.WrapError(err)
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "webmasters.sites.add", "response", internallog.HTTPResponse(res, nil))
	return nil
}

type SitesDeleteCall struct {
	s          *Service
	siteUrl    string
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Delete:  Removes a site from the set of the user's Search Console sites.
//
//   - siteUrl: The URI of the property as defined in Search Console.
//     **Examples:** `http://www.example.com/` or `sc-domain:example.com`.
func (r *SitesService) Delete(siteUrl string) *SitesDeleteCall {
	c := &SitesDeleteCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.siteUrl = siteUrl
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *SitesDeleteCall) Fields(s ...googleapi.Field) *SitesDeleteCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *SitesDeleteCall) Context(ctx context.Context) *SitesDeleteCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *SitesDeleteCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *SitesDeleteCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "webmasters/v3/sites/{siteUrl}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("DELETE", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"siteUrl": c.siteUrl,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "webmasters.sites.delete", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "webmasters.sites.delete" call.
func (c *SitesDeleteCall) Do(opts ...googleapi.CallOption) error {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if err != nil {
		return err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return gensupport.WrapError(err)
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "webmasters.sites.delete", "response", internallog.HTTPResponse(res, nil))
	return nil
}

type SitesGetCall struct {
	s            *Service
	siteUrl      string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get:  Retrieves information about specific site.
//
//   - siteUrl: The URI of the property as defined in Search Console.
//     **Examples:** `http://www.example.com/` or `sc-domain:example.com`.
func (r *SitesService) Get(siteUrl string) *SitesGetCall {
	c := &SitesGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.siteUrl = siteUrl
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *SitesGetCall) Fields(s ...googleapi.Field) *SitesGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *SitesGetCall) IfNoneMatch(entityTag string) *SitesGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *SitesGetCall) Context(ctx context.Context) *SitesGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *SitesGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *SitesGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "webmasters/v3/sites/{siteUrl}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"siteUrl": c.siteUrl,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "webmasters.sites.get", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "webmasters.sites.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *WmxSite.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *SitesGetCall) Do(opts ...googleapi.CallOption) (*WmxSite, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &WmxSite{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "webmasters.sites.get", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type SitesListCall struct {
	s            *Service
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List:  Lists the user's Search Console sites.
func (r *SitesService) List() *SitesListCall {
	c := &SitesListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *SitesListCall) Fields(s ...googleapi.Field) *SitesListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *SitesListCall) IfNoneMatch(entityTag string) *SitesListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *SitesListCall) Context(ctx context.Context) *SitesListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *SitesListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *SitesListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "webmasters/v3/sites")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "webmasters.sites.list", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "webmasters.sites.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *SitesListResponse.ServerResponse.Header or (if a response was returned at
// all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *SitesListCall) Do(opts ...googleapi.CallOption) (*SitesListResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &SitesListResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "webmasters.sites.list", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type UrlInspectionIndexInspectCall struct {
	s                      *Service
	inspecturlindexrequest *InspectUrlIndexRequest
	urlParams_             gensupport.URLParams
	ctx_                   context.Context
	header_                http.Header
}

// Inspect: Index inspection.
func (r *UrlInspectionIndexService) Inspect(inspecturlindexrequest *InspectUrlIndexRequest) *UrlInspectionIndexInspectCall {
	c := &UrlInspectionIndexInspectCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.inspecturlindexrequest = inspecturlindexrequest
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *UrlInspectionIndexInspectCall) Fields(s ...googleapi.Field) *UrlInspectionIndexInspectCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *UrlInspectionIndexInspectCall) Context(ctx context.Context) *UrlInspectionIndexInspectCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *UrlInspectionIndexInspectCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *UrlInspectionIndexInspectCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.inspecturlindexrequest)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/urlInspection/index:inspect")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "searchconsole.urlInspection.index.inspect", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "searchconsole.urlInspection.index.inspect" call.
// Any non-2xx status code is an error. Response headers are in either
// *InspectUrlIndexResponse.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *UrlInspectionIndexInspectCall) Do(opts ...googleapi.CallOption) (*InspectUrlIndexResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &InspectUrlIndexResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "searchconsole.urlInspection.index.inspect", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type UrlTestingToolsMobileFriendlyTestRunCall struct {
	s                            *Service
	runmobilefriendlytestrequest *RunMobileFriendlyTestRequest
	urlParams_                   gensupport.URLParams
	ctx_                         context.Context
	header_                      http.Header
}

// Run: Runs Mobile-Friendly Test for a given URL.
func (r *UrlTestingToolsMobileFriendlyTestService) Run(runmobilefriendlytestrequest *RunMobileFriendlyTestRequest) *UrlTestingToolsMobileFriendlyTestRunCall {
	c := &UrlTestingToolsMobileFriendlyTestRunCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.runmobilefriendlytestrequest = runmobilefriendlytestrequest
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *UrlTestingToolsMobileFriendlyTestRunCall) Fields(s ...googleapi.Field) *UrlTestingToolsMobileFriendlyTestRunCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *UrlTestingToolsMobileFriendlyTestRunCall) Context(ctx context.Context) *UrlTestingToolsMobileFriendlyTestRunCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *UrlTestingToolsMobileFriendlyTestRunCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *UrlTestingToolsMobileFriendlyTestRunCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.runmobilefriendlytestrequest)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/urlTestingTools/mobileFriendlyTest:run")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "searchconsole.urlTestingTools.mobileFriendlyTest.run", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "searchconsole.urlTestingTools.mobileFriendlyTest.run" call.
// Any non-2xx status code is an error. Response headers are in either
// *RunMobileFriendlyTestResponse.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *UrlTestingToolsMobileFriendlyTestRunCall) Do(opts ...googleapi.CallOption) (*RunMobileFriendlyTestResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &RunMobileFriendlyTestResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "searchconsole.urlTestingTools.mobileFriendlyTest.run", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}
